<?php
namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Models\DataKlien;
use App\Models\Proyek;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class KlienController extends Controller
{
    public function index(Request $request)
    {
        $query = DataKlien::with('proyek', 'user');
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search != '') {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('alamat', 'like', '%' . $searchTerm . '%')
                  ->orWhere('jenis_bangunan', 'like', '%' . $searchTerm . '%')
                  ->orWhereHas('proyek', function($query) use ($searchTerm) {
                      $query->where('nama_proyek', 'like', '%' . $searchTerm . '%');
                  })
                  ->orWhereHas('user', function($query) use ($searchTerm) {
                      $query->where('name', 'like', '%' . $searchTerm . '%')
                            ->orWhere('username', 'like', '%' . $searchTerm . '%');
                  });
            });
        }

        // Filter berdasarkan Status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        
        $dataKliens = $query->latest()->get();
        $proyeks = Proyek::all();
        $klienUsers = User::where('role', 'user')->get();
        $globalSettings = Setting::pluck('value', 'key')->all();
        
        // Mendapatkan daftar jenis bangunan unik untuk filter
        $jenisBangunanList = DataKlien::distinct()->pluck('jenis_bangunan')->filter();
        
        // --- [DIHAPUS] Logika untuk Kanban Task sudah tidak ada ---
        
        return view('klien.index', compact(
            'dataKliens', 
            'proyeks', 
            'klienUsers',
            'globalSettings', 
            'jenisBangunanList'
        ));
    }
    /**
     * Menampilkan form untuk membuat data klien baru.
     */
    public function create()
    {
        $proyeks = Proyek::whereDoesntHave('dataKlien')->get();
        $users = User::where('role', 'user')->get();
        $globalSettings = Setting::pluck('value', 'key')->all();

        return view('klien.create', compact('proyeks', 'users', 'globalSettings'));
    }

    /**
     * Menyimpan data klien baru ke database.
     */
    public function store(Request $request)
    {
        // ... (Logika store Anda sudah OK)
        $validator = Validator::make($request->all(), [
            'id_proyek' => 'required|exists:proyeks,id_proyek|unique:data_kliens,id_proyek',
            'id_user' => 'required|exists:users,id',
            'alamat' => 'required|string',
            'jenis_bangunan' => 'required|string|max:100',
            'ktp' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'shm' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'gambar' => 'nullable|array', 
            'gambar.*.file' => 'nullable|file|mimes:jpg,jpeg,png,pdf,dwg,dxf|max:5120',
            'gambar.*.deskripsi' => 'nullable|string|max:255',
            'ITR' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'perhitungan_MEP' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $request->except(['ktp', 'shm', 'gambar', 'ITR', 'perhitungan_MEP']);
        $uploadPath = 'public/dokumen_klien/' . $request->id_proyek;

        // Proses Upload File Single
        $fileFields = ['ktp', 'shm', 'ITR', 'perhitungan_MEP'];
        foreach ($fileFields as $field) {
            if ($request->hasFile($field)) {
                $path = $request->file($field)->store($uploadPath);
                $data[$field] = $path;
            }
        }

        // Proses Upload Multiple 'gambar' dengan deskripsi
        if ($request->has('gambar')) {
            $gambarData = [];
            foreach ($request->gambar as $item) {
                if (isset($item['file']) && $item['file']->isValid()) {
                    $path = $item['file']->store($uploadPath);
                    $gambarData[] = [
                        'path' => $path,
                        'deskripsi' => $item['deskripsi'] ?? null
                    ];
                }
            }
            $data['gambar'] = $gambarData; 
        }

        DataKlien::create($data);

        return redirect()->route('klien.index')->with('success', 'Data Klien berhasil ditambahkan.');
    }

    /**
     * Menampilkan detail spesifik data klien.
     */
    public function show($id)
    {
        // ... (Logika show Anda sudah OK)
        $dataKlien = DataKlien::with('proyek', 'user')->findOrFail($id);
        $globalSettings = Setting::pluck('value', 'key')->all();
        
        return view('klien.show', compact('dataKlien', 'globalSettings'));
    }

    /**
     * Menampilkan form untuk mengedit data klien.
     */
    public function edit($id)
    {
        // ... (Logika edit Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);
        $proyeks = Proyek::all();
        $users = User::where('role', 'user')->get();
        $globalSettings = Setting::pluck('value', 'key')->all();
        
        return view('klien.edit', compact('dataKlien', 'proyeks', 'users', 'globalSettings'));
    }

    /**
     * Update data klien di database.
     */
    public function update(Request $request, $id)
    {
        // ... (Logika update Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'id_proyek' => [
                'required',
                'exists:proyeks,id_proyek',
                Rule::unique('data_kliens', 'id_proyek')->ignore($dataKlien->id),
            ],
            'id_user' => 'required|exists:users,id',
            'alamat' => 'required|string',
            'jenis_bangunan' => 'required|string|max:100',
            'ktp' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'shm' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'gambar' => 'nullable|array', 
            'gambar.*.file' => 'nullable|file|mimes:jpg,jpeg,png,pdf,dwg,dxf|max:5120', 
            'gambar.*.deskripsi' => 'nullable|string|max:255',
            'ITR' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'perhitungan_MEP' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $request->except(['_token', '_method', 'ktp', 'shm', 'gambar', 'ITR', 'perhitungan_MEP']);
        $uploadPath = 'public/dokumen_klien/' . $request->id_proyek;

        // Update File Single
        $fileFields = ['ktp', 'shm', 'ITR', 'perhitungan_MEP'];
        foreach ($fileFields as $field) {
            if ($request->hasFile($field)) {
                if ($dataKlien->{$field}) {
                    Storage::delete($dataKlien->{$field});
                }
                $path = $request->file($field)->store($uploadPath);
                $data[$field] = $path;
            }
        }
        
        // Logika Update Multiple 'gambar' (Replace)
        if ($request->has('gambar')) {
            // Hapus semua file gambar lama
            if (is_array($dataKlien->gambar)) {
                foreach ($dataKlien->gambar as $oldItem) {
                    if(isset($oldItem['path'])) {
                        Storage::delete($oldItem['path']);
                    }
                }
            }

            // Upload semua file gambar baru
            $gambarData = [];
            foreach ($request->gambar as $item) {
                if (isset($item['file']) && $item['file']->isValid()) {
                    $path = $item['file']->store($uploadPath);
                    $gambarData[] = [
                        'path' => $path,
                        'deskripsi' => $item['deskripsi'] ?? null
                    ];
                }
            }
            $data['gambar'] = $gambarData;
        }

        $dataKlien->update($data);

        return redirect()->route('klien.index')->with('success', 'Data Klien berhasil diperbarui.');
    }

    /**
     * Menghapus data klien dari database.
     */
    public function destroy($id)
    {
        // ... (Logika destroy Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);

        // Hapus file single
        $fileFields = ['ktp', 'shm', 'ITR', 'perhitungan_MEP'];
        foreach ($fileFields as $field) {
            if ($dataKlien->{$field}) {
                Storage::delete($dataKlien->{$field});
            }
        }
        
        // Hapus semua file dari array 'gambar'
        if (is_array($dataKlien->gambar)) {
            foreach ($dataKlien->gambar as $item) {
                 if(isset($item['path'])) {
                    Storage::delete($item['path']);
                }
            }
        }
        
        $dataKlien->delete();

        return redirect()->route('klien.index')->with('success', 'Data Klien berhasil dihapus.');
    }

    /**
     * Fungsi untuk mendownload file yang di-upload.
     */
    public function downloadFile($id, $field)
    {
        // ... (Logika downloadFile Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);

        $fileFields = ['ktp', 'shm', 'ITR', 'perhitungan_MEP'];
        if (!in_array($field, $fileFields) || !$dataKlien->{$field}) {
            abort(404, 'File not found.');
        }

        $filePath = $dataKlien->{$field};

        if (!Storage::exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::download($filePath);
    }
    
    /**
     * Fungsi untuk mendownload file gambar dari array gambar.
     */
    public function downloadGambar($id, $index)
    {
        // ... (Logika downloadGambar Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);
        
        if (!is_array($dataKlien->gambar) || !isset($dataKlien->gambar[$index])) {
            abort(404, 'Gambar not found.');
        }
        
        $gambar = $dataKlien->gambar[$index];
        $filePath = $gambar['path'];
        
        if (!Storage::exists($filePath)) {
            abort(404, 'File not found.');
        }
        
        // Dapatkan nama file asli
        $fileName = basename($filePath);
        $deskripsi = $gambar['deskripsi'] ?? 'Gambar';
        
        // Buat nama file yang lebih deskriptif
        $downloadName = $deskripsi . '_' . $fileName;
        
        return Storage::download($filePath, $downloadName);
    }
    
    /**
     * Fungsi untuk menghapus satu gambar dari array gambar.
     */
    public function hapusGambar($id, $index)
    {
        // ... (Logika hapusGambar Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);
        
        if (!is_array($dataKlien->gambar) || !isset($dataKlien->gambar[$index])) {
            return redirect()->back()->with('error', 'Gambar tidak ditemukan.');
        }
        
        $gambar = $dataKlien->gambar[$index];
        $filePath = $gambar['path'];
        
        // Hapus file fisik
        if (Storage::exists($filePath)) {
            Storage::delete($filePath);
        }
        
        // Hapus dari array
        $gambarArray = $dataKlien->gambar;
        unset($gambarArray[$index]);
        
        // Re-index array
        $gambarArray = array_values($gambarArray);
        
        // Update database
        $dataKlien->gambar = $gambarArray;
        $dataKlien->save();
        
        return redirect()->back()->with('success', 'Gambar berhasil dihapus.');
    }
    
    /**
     * Fungsi untuk menampilkan preview gambar.
     */
    public function previewGambar($id, $index)
    {
        // ... (Logika previewGambar Anda sudah OK)
        $dataKlien = DataKlien::findOrFail($id);
        
        if (!is_array($dataKlien->gambar) || !isset($dataKlien->gambar[$index])) {
            abort(404, 'Gambar not found.');
        }
        
        $gambar = $dataKlien->gambar[$index];
        $filePath = $gambar['path'];
        
        if (!Storage::exists($filePath)) {
            abort(404, 'File not found.');
        }
        
        $file = Storage::get($filePath);
        $mimeType = Storage::mimeType($filePath);
        
        return response($file, 200)->header('Content-Type', $mimeType);
    }
    
    /**
     * Fungsi untuk mendapatkan daftar proyek yang tersedia (untuk AJAX).
     */
    public function getAvailableProyeks()
    {
        // ... (Logika getAvailableProyeks Anda sudah OK)
        $proyeks = Proyek::whereDoesntHave('dataKlien')->get(['id_proyek', 'nama_proyek']);
        
        return response()->json($proyeks);
    }
}